<?php
// Retorna JSON SEMPRE
header('Content-Type: application/json; charset=utf-8');
// Evita cache
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

require __DIR__ . '/config.php';
session_start();

// Placa vem da sessão (fluxo normal) ou via GET (teste)
$placa = $_SESSION['placa'] ?? ($_GET['placa'] ?? '');
$placa = strtoupper(trim((string)$placa));

if ($placa === '') {
    http_response_code(400);
    echo json_encode(['erro' => 'Placa não informada.']);
    exit;
}

// Chama a API externa
$dados = chamarApiPosicaoFila($placa);
if (!$dados) {
    http_response_code(502);
    echo json_encode(['erro' => 'Falha na comunicação com a API.']);
    exit;
}

/**
 * Gatilho "foi chamado": data_chamada_descarga e hora_chamada_descarga preenchidos.
 * - Envia Web Push para a placa (uma vez por evento).
 * - Envia WhatsApp para o celular do motorista (uma vez por evento), se houver e for válido.
 * Evento é identificado por ID: "PLACA|DATA|HORA(HH:MM)".
 */
try {
    $veiculo = $dados['dados']['dadosVeiculo'] ?? null;

    if ($veiculo) {
        $dataCh = $veiculo['data_chamada_descarga'] ?? null;
        $horaCh = $veiculo['hora_chamada_descarga'] ?? null;

        if (!empty($dataCh) && !empty($horaCh)) {
            $placaEnvio = strtoupper(trim((string)($veiculo['veiculo_fila_descarga'] ?? $placa)));
            $horaMin    = substr((string)$horaCh, 0, 5);
            $id         = $placaEnvio . '|' . $dataCh . '|' . $horaMin;

            // ====================== WEB PUSH (dedupe notificados.json) ======================
            $fileNot = __DIR__ . '/notificados.json';
            $notificados = [];
            if (is_file($fileNot)) {
                $tmp = json_decode(@file_get_contents($fileNot), true);
                if (is_array($tmp)) $notificados = $tmp;
            }

            if (empty($notificados[$id])) {
                $titulo   = '🚚 Seu caminhão foi chamado';
                $mensagem = sprintf(
                    'Placa %s chamada em %s às %s.',
                    $placaEnvio,
                    formatarDataBr($dataCh),
                    $horaMin
                );

                enviarPushPlaca($placaEnvio, $titulo, $mensagem, [
                    'placa' => $placaEnvio,
                    'data'  => $dataCh,
                    'hora'  => $horaMin,
                    'url'   => '/resultado',
                ]);

                // Marca PUSH como enviado
                $notificados[$id] = ['quando' => date('c')];
                $tmpFile = $fileNot . '.tmp';
                @file_put_contents($tmpFile, json_encode($notificados, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                @rename($tmpFile, $fileNot);
            }

            // ====================== WHATSAPP (dedupe zapi_notificados.json) ======================
            $telRaw      = $veiculo['celular_motorista'] ?? null;
            $numeroE164  = normalizarTelefoneBr($telRaw, 'inserir9'); // ou 'estrito'
            if ($numeroE164) {
                $whFile = __DIR__ . '/zapi_notificados.json';
                $whNot  = [];
                if (is_file($whFile)) {
                    $tmp = json_decode(@file_get_contents($whFile), true);
                    if (is_array($tmp)) $whNot = $tmp;
                }

                if (empty($whNot[$id])) {
                            $mensagemWhats = sprintf(
            "ATENÇÃO: seu caminhão placa %s foi CHAMADO para descarregar em %s às %s,\nDirija-se à balança imediatamente ⚠️\n\nMensagem oficial do GRUPO ATT",
            $placaEnvio,
            formatarDataBr($dataCh),
            $horaMin,
            (string)($veiculo['nome_empresa'] ?? '-')
        );

                    $ok = enviarWhatsApp($numeroE164, $mensagemWhats);

                    if ($ok) {
                        $whNot[$id] = [
                            'quando'   => date('c'),
                            'placa'    => $placaEnvio,
                            'telefone' => $numeroE164,
                        ];
                        $tmpFile = $whFile . '.tmp';
                        @file_put_contents($tmpFile, json_encode($whNot, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                        @rename($tmpFile, $whFile);
                    }
                }
            }
        }
    }
} catch (\Throwable $e) {
    // Não quebra a resposta para o front; faça log para diagnóstico
    error_log('Erro no gatilho de chamada: ' . $e->getMessage());
}


// Resposta crua da API (mantém compatibilidade com o front)
echo json_encode($dados, JSON_UNESCAPED_UNICODE);