<?php
session_start();
require __DIR__ . '/config.php';

/* ===== Rotas utilitárias ===== */
if (isset($_GET['logout'])) {
  unset($_SESSION['placa']);
  setcookie('placa', '', time() - 3600, '/');
  session_destroy();
  header("Location: inicio");
  exit;
}

if (isset($_GET['forceExit'])) {
  $_SESSION['flash_erro'] = "Seu caminhão não está mais na fila.";
  unset($_SESSION['placa']);
  setcookie('placa', '', time() - 3600, '/');
  header("Location: inicio");
  exit;
}

if (isset($_GET['apiFail'])) {
  $_SESSION['flash_erro'] = "Falha na comunicação com a API. Tente novamente.";
  header("Location: inicio");
  exit;
}

/* ===== Entrada padrão ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $placa = strtoupper(trim($_POST['placa'] ?? ''));

  if ($placa === '') {
    $_SESSION['flash_erro'] = "Informe a placa.";
    header("Location: inicio");
    exit;
  }

  if (!preg_match('/^[A-Z0-9]{7,8}$/', $placa)) {
    $_SESSION['flash_erro'] = "Placa incorreta. Use apenas letras e números (7 ou 8 caracteres).";
    header("Location: inicio");
    exit;
  }

  // 🔒 Exige aceite das diretrizes
  if (empty($_POST['diretrizes'])) {
    $_SESSION['flash_erro'] = "Você precisa aceitar o Documento de Diretrizes para continuar.";
    header("Location: inicio");
    exit;
  }

  // 📝 LOG DE ACEITE (arquivo por dia, 1 linha JSON por registro)
  $versao = $_POST['diretrizes_versao'] ?? '1.0';
  $ip     = $_SERVER['REMOTE_ADDR'] ?? 'desconhecido';
  $ua     = substr($_SERVER['HTTP_USER_AGENT'] ?? 'desconhecido', 0, 300);
  $quando = date('c'); // ISO 8601

  $logDir  = __DIR__ . '/logs';
  if (!is_dir($logDir)) {
    @mkdir($logDir, 0755, true);
  }
  $logFile = $logDir . '/aceites_' . date('Y-m-d') . '.log';

  $entry = json_encode([
      'timestamp' => $quando,
      'placa'     => $placa,
      'ip'        => $ip,
      'ua'        => $ua,
      'versao'    => $versao,
      'aceitou'   => true
    ], JSON_UNESCAPED_UNICODE) . PHP_EOL;

  $ok = @file_put_contents($logFile, $entry, FILE_APPEND | LOCK_EX);
  if ($ok === false) {
    // Se quiser, trate erro de escrita em log (mas não bloqueie o usuário)
    error_log("Falha ao gravar log de aceite em $logFile");
  }

  // Sessão + cookie (30 dias)
  $_SESSION['placa'] = $placa;
  setcookie('placa', $placa, time() + 60*60*24*30, '/');

} else {
  // Sessão pode ter expirado: restaura do cookie
  if (empty($_SESSION['placa']) && !empty($_COOKIE['placa'])) {
    $_SESSION['placa'] = strtoupper($_COOKIE['placa']);
  }

  if (empty($_SESSION['placa'])) {
    header("Location: inicio");
    exit;
  }
}



$placa = $_SESSION['placa'];
?>
<!doctype html>
<html lang="pt-br">
<head>
  <meta charset="utf-8">
  <title>Consulta de Fila</title>
  <meta name="viewport" content="width=device-width, initial-scale=1, viewport-fit=cover">
  <meta name="theme-color" content="#0b2d55">
  <link rel="stylesheet" href="assets/style.css">
  <link rel="stylesheet" href="assets/consulta.css">
  <link rel="icon" type="image/png" href="assets/Logo-ATT2.png">
  <style>
    body {
      padding-top: env(safe-area-inset-top);
      padding-bottom: env(safe-area-inset-bottom);
      background-color: #fff;
    }
    .container {
      min-height: calc(100vh - env(safe-area-inset-top) - env(safe-area-inset-bottom));
      display: flex;
      flex-direction: column;
      justify-content: space-between;
    }
  </style>
</head>
<body>
  <div class="container">

    <div>
      <div class="header">
        <a class="btn" href="?logout=1">← Voltar</a>
        <span class="badge wait" id="statusBadge">Carregando...</span>
      </div>

      <div class="titulo" id="tituloCaminhao" style="text-align:center; width:100%;">
        📋 Caminhão <?= htmlspecialchars($placa) ?>
      </div>

      <p class="subtitulo" id="subInfo"></p>
      <div class="info-grid" id="cards"></div>
    </div>

    <div class="footer">© <?= date('Y') ?> Todos os direitos reservados Grupo ATT</div>
  </div>

  <!-- Script de atualização da fila -->
  <script>
    const cardsEl   = document.getElementById('cards');
    const tituloEl  = document.getElementById('tituloCaminhao');
    const subInfoEl = document.getElementById('subInfo');
    const statusBadge = document.getElementById('statusBadge');

    function hhmm(horaCompleta){
      if(!horaCompleta) return "-";
      const partes = horaCompleta.split(':').slice(0,2);
      return partes.join(':');
    }

    function dataBr(yyyy_mm_dd){
      if(!yyyy_mm_dd) return "-";
      const [y,m,d] = yyyy_mm_dd.split('-');
      return `${d.padStart(2,'0')}/${m.padStart(2,'0')}/${y}`;
    }

    function montarCard(label, valor, alerta=false){
      const div = document.createElement('div');
      div.className = 'card' + (alerta ? ' card-alerta' : '');
      div.innerHTML = `<div class="label"><strong>${label}</strong></div><div class="valor">${valor}</div>`;
      return div;
    }

    async function carregar(){
      try{
        statusBadge.textContent = 'Atualizando...';

        const r = await fetch('api_consulta.php?_=' + Date.now(), { cache:'no-store' });
        const data = await r.json();

        if (data.erro){
          window.location.href = "resultado?apiFail=1";
          return;
        }

        const veiculo = (data.dados && data.dados.dadosVeiculo) ? data.dados.dadosVeiculo : null;

        if (!veiculo) {
          // Saiu da fila -> aciona rota que limpa cookie e sessão
          window.location.href = "resultado?forceExit=1";
          return;
        }

        const foiChamado = !!(veiculo.data_chamada_descarga && veiculo.hora_chamada_descarga);

        tituloEl.textContent = `📋 Caminhão ${veiculo.veiculo_fila_descarga}`;

        statusBadge.className = 'badge ' + (foiChamado ? 'success' : 'wait');
        statusBadge.textContent = foiChamado ? 'Chamado' : 'Aguardando';

        const infos = [
          { label: "Sua Posição na Fila", value: veiculo.ordem },
          { label: "Empresa", value: veiculo.nome_empresa },
          { label: "Produto", value: veiculo.produto || "-" },
          { label: "Placa do seu Caminhão", value: veiculo.veiculo_fila_descarga },
          ...(veiculo.placa_anterior ? [{ label: "Placa do caminhão à sua frente", value: veiculo.placa_anterior }] : []),
          { label: "Número do Contrato", value: veiculo.contrato_fila_descarga },
          {
            label: "Horário que entrou na fila",
            value: (veiculo.data_fila_descarga && veiculo.hora_fila_descarga)
              ? `${dataBr(veiculo.data_fila_descarga)} às ${hhmm(veiculo.hora_fila_descarga)}`
              : "-",
          },
          {
            label: "Horário que foi chamado",
            value: (veiculo.data_chamada_descarga && veiculo.hora_chamada_descarga)
              ? `${dataBr(veiculo.data_chamada_descarga)} às ${hhmm(veiculo.hora_chamada_descarga)}`
              : "Aguardando ser chamado",
            alerta: (veiculo.data_chamada_descarga && veiculo.hora_chamada_descarga)
          },
        ];

        cardsEl.innerHTML = '';
        for (const item of infos){
          cardsEl.appendChild(montarCard(item.label, item.value, !!item.alerta && item.label === 'Horário que foi chamado'));
        }

        subInfoEl.textContent = '';
      } catch(e){
        window.location.href = "resultado?apiFail=1";
      }
    }

    // primeira carga + atualização automática (4 minutos)
    carregar();
    setInterval(carregar, 120000);
  </script>

  <!-- Script de registro de push -->
  <script>
  async function registrarPush(placa){
    if (!('serviceWorker' in navigator) || !('PushManager' in window)) return;

    try {
      // registra o SW dentro da própria pasta do sistema
      const reg = await navigator.serviceWorker.register('./service-worker.js', { scope: './' });

      // solicita permissão
      const perm = await Notification.requestPermission();
      if (perm !== 'granted') return;

      // busca chave pública VAPID (arquivo na mesma pasta)
      const r = await fetch('webpush_public_key.php', { cache: 'no-store' });
      const { publicKey } = await r.json();

      // cria inscrição
      const sub = await reg.pushManager.subscribe({
        userVisibleOnly: true,
        applicationServerKey: urlBase64ToUint8Array(publicKey),
      });

      // salva inscrição (arquivo na mesma pasta)
      await fetch('salvar_inscricao.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ placa, subscription: sub }),
      });
    } catch (e) {
      console.error('WebPush registrar erro:', e);
    }
  }

  function urlBase64ToUint8Array(base64){
    const pad = '='.repeat((4 - base64.length % 4) % 4);
    const b64 = (base64 + pad).replace(/-/g, '+').replace(/_/g, '/');
    const raw = atob(b64);
    const out = new Uint8Array(raw.length);
    for (let i = 0; i < raw.length; i++) out[i] = raw.charCodeAt(i);
    return out;
  }

  // chama o registro passando a placa da sessão
  registrarPush("<?= htmlspecialchars($placa) ?>");
  </script>
</body>
</html>
