<?php
// salvar_inscricao.php
declare(strict_types=1);

require __DIR__ . '/config.php';
header('Content-Type: application/json; charset=utf-8');

// Aceita apenas POST com JSON
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  echo json_encode(['ok' => false, 'msg' => 'Método não permitido']);
  exit;
}

$raw = file_get_contents('php://input') ?: '';
$body = json_decode($raw, true);
if (!is_array($body)) {
  http_response_code(400);
  echo json_encode(['ok' => false, 'msg' => 'JSON inválido']);
  exit;
}

// Dados obrigatórios
$placa = strtoupper(trim((string)($body['placa'] ?? '')));
$subscription = $body['subscription'] ?? null;
$endpoint = is_array($subscription) ? ($subscription['endpoint'] ?? '') : '';

if ($placa === '' || !$subscription || !is_string($endpoint) || $endpoint === '') {
  http_response_code(400);
  echo json_encode(['ok' => false, 'msg' => 'Dados inválidos (placa/subscription/endpoint)']);
  exit;
}

// Opcional: limitar tamanho da placa (7-8) e caracteres
if (!preg_match('/^[A-Z0-9]{7,8}$/', $placa)) {
  http_response_code(400);
  echo json_encode(['ok' => false, 'msg' => 'Placa inválida']);
  exit;
}

$arquivo = __DIR__ . '/subs.json';

// Carrega subs com tolerância
$subs = [];
if (file_exists($arquivo)) {
  $json = file_get_contents($arquivo);
  $tmp  = json_decode($json, true);
  if (is_array($tmp)) $subs = $tmp;
}

// Garante estrutura da placa
if (!isset($subs[$placa]) || !is_array($subs[$placa])) {
  $subs[$placa] = [];
}

// Atualiza OU adiciona por endpoint (sem duplicar)
$atualizado = false;
foreach ($subs[$placa] as $i => $s) {
  if (($s['endpoint'] ?? '') === $endpoint) {
    // Atualiza keys/expiração se mudou
    $subs[$placa][$i] = $subscription;
    $subs[$placa][$i]['_meta'] = [
      'ua'   => $_SERVER['HTTP_USER_AGENT'] ?? '',
      'ip'   => $_SERVER['REMOTE_ADDR'] ?? '',
      'ts'   => date('c'),
    ];
    $atualizado = true;
    break;
  }
}

if (!$atualizado) {
  // Opcional: limitar a no máx. N inscrições por placa (ex.: 10)
  if (count($subs[$placa]) >= 10) {
    // remove a mais antiga
    array_shift($subs[$placa]);
  }
  $subscription['_meta'] = [
    'ua'   => $_SERVER['HTTP_USER_AGENT'] ?? '',
    'ip'   => $_SERVER['REMOTE_ADDR'] ?? '',
    'ts'   => date('c'),
  ];
  $subs[$placa][] = $subscription;
}

// Escrita atômica (tmp + rename) pra evitar arquivo corrompido
$tmpFile = $arquivo . '.tmp';
$ok = (bool)file_put_contents($tmpFile, json_encode($subs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
if ($ok) {
  $ok = @rename($tmpFile, $arquivo);
}
if (!$ok) {
  // fallback direto (menos seguro, mas evita falha total)
  @unlink($tmpFile);
  $ok = (bool)file_put_contents($arquivo, json_encode($subs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

if (!$ok) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'msg' => 'Falha ao salvar inscrição (permissões de escrita?)']);
  exit;
}

echo json_encode([
  'ok'      => true,
  'msg'     => $atualizado ? 'Inscrição atualizada' : 'Inscrição salva',
  'placa'   => $placa,
  'count'   => count($subs[$placa]),
]);
